--------------------------------------------------------------------
-- Excel Import
--------------------------------------------------------------------

-- Basisfunktion für Excel Import
CREATE OR REPLACE FUNCTION api.import_excel_via_function(
    excel_file bytea,
    import_func_name text,
    field_mapping json
    )
    RETURNS TEXT
    AS $$
    import io
    import openpyxl
    import json

    try:
        # JSON-Mapping aus Parameter extrahieren (sicher!)
        mapping_param = json.loads(field_mapping)

        file = io.BytesIO(excel_file)
        wb = openpyxl.load_workbook(file, data_only=True)
        ws = wb.active

        headers = [
            cell.value.strip() if isinstance(cell.value, str) else str(cell.value)
            for cell in next(ws.iter_rows(min_row=1, max_row=1))
        ]

        rows_inserted = 0

        plan = plpy.prepare(
            f"SELECT {import_func_name}($1::json)",
            ["json"]
        )

        for row in ws.iter_rows(min_row=2, values_only=True):
            excel_row = dict(zip(headers, row))
            mapped = {}

            for db_field, config in mapping_param.items():
                source = config.get("source")
                value = config.get("value")
                if source == "column":
                    mapped[db_field] = excel_row.get(value)
                elif source == "fixed":
                    mapped[db_field] = value
                else:
                    raise Exception(f"Ungültige Quelle '{source}' für Feld '{db_field}'")

            json_row = json.dumps(mapped)
            plpy.execute(plan, [json_row])
            rows_inserted += 1

        return f"{rows_inserted} Zeilen importiert über {import_func_name}."
    except Exception as e:
        return f"Fehler beim Import: {str(e)}"
    $$ LANGUAGE plpython3u;
--

/* Projekte Einlesen */

-- Projekte via Excel einlesen x_950_import.projekt verwenden
-- Vereinfachungen aufgrund Struktur bei Firma Emis (PAL) siehe value Zuweisung
CREATE OR REPLACE FUNCTION api.import_projekt_row_call(
    json_row json
    )
    RETURNS void
    AS $$
    BEGIN
      -- Pflichtfeldprüfung
      IF json_row->>'pr_an_nr' IS NULL THEN
        RAISE EXCEPTION 'Fehlende Pflichtspalte: pr_an_nr';
      END IF;

      -- Direkter Insert in die Zieltabelle mit Datenaufbereitung
      INSERT INTO x_950_import.projekt (
        pr_parent_ab_ap_nr,
        pr_an_nr,
        pr_an_bez,
        pr_ab_ap_nr,
        pr_ab_ap_bem,
        pr_a2_ks,
        pr_a2_subject
      )
      VALUES (
        upper(json_row->>'pr_an_nr'),       -- pr_parent_ab_ap_nr
        upper(json_row->>'pr_an_nr'),       -- pr_an_nr
        json_row->>'pr_an_bez',
        upper(json_row->>'pr_ab_ap_nr'),
        json_row->>'pr_ab_ap_bem',
        upper(json_row->>'pr_a2_ks'),
        json_row->>'pr_a2_subject'
      );
    END $$ LANGUAGE plpgsql;


-- always-show-login für body zeigt Anmeldeinformationen immer
CREATE OR REPLACE FUNCTION api.importform()
RETURNS "text/html"
AS $$
SELECT concat_ws(E'\n',
  '<!DOCTYPE html>',
  '<html lang="de">',
  api.html_head_get('PRODAT Projekt Import'),
  $html$
  <body class="bg-gray-100 text-gray-900 min-h-screen">
        <!-- Navigation -->
    <nav id="mainNavbar" class="w-full z-50 bg-white shadow px-6 py-2 sm:py-4 flex justify-between items-center">
      <div>
          <div class="flex items-center gap-2">
            <!-- Tabellen-Icon (Excel-ähnlich) -->
            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <rect x="3" y="4" width="18" height="16" rx="2" ry="2" />
              <line x1="3" y1="10" x2="21" y2="10" />
              <line x1="9" y1="4" x2="9" y2="20" />
              <line x1="15" y1="4" x2="15" y2="20" />
            </svg>

            <!-- Titel -->
            <h1 class="text-xl font-semibold tracking-tight">Excel Import</h1>
          </div>
          <span class="ml-9 text-xs text-gray-400 font-semibold tracking-wide">PRODAT ERP</span>
        </div>

      <div class="flex items-center gap-4">
        <!-- User Dropdown Trigger -->
        <div id="userDropdownWrap" class="relative hidden" style="display: none;">
          <button id="userMenuBtn" type="button" class="flex items-center gap-2 px-2 py-1 rounded-xl hover:bg-gray-100 focus:outline-none transition">
            <!-- Avatar Initialen -->
            <div id="userAvatar" class="w-9 h-9 flex items-center justify-center bg-gradient-to-tr from-green-400 to-green-700 text-white text-lg font-bold rounded-full shadow-sm"></div>
            <!-- Username & Badge -->
            <div class="flex flex-col items-start">
              <span id="userName" class="text-base font-semibold text-gray-800"></span>
              <span id="adminBadge" class="flex items-center gap-1 text-xs font-medium bg-green-100 text-green-700 px-2 py-0.5 rounded">
                <svg class="w-4 h-4 text-green-600" fill="none" stroke="currentColor" stroke-width="3" viewBox="0 0 24 24">
                  <title>Admin-Berechtigung</title>
                  <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                </svg>
                Admin
              </span>
            </div>
            <svg class="w-4 h-4 ml-1 text-gray-400 transition" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" d="M19 9l-7 7-7-7"></path>
            </svg>
          </button>
          <!-- Dropdown-Menu -->
          <div id="userDropdown" class="z-30 absolute right-0 mt-2 w-44 bg-white rounded-xl shadow-lg border border-gray-100 opacity-0 pointer-events-none scale-95 transition-all duration-200">
            <div class="py-2">
              <button id="logoutDropdownBtn" class="w-full flex items-center px-4 py-2 text-sm text-red-600 hover:bg-red-50 rounded-b-xl transition">
                <svg class="w-4 h-4 mr-2 text-red-400" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1m0-9V7" />
                </svg>
                Abmelden
              </button>
            </div>
          </div>
        </div>
        <button id="loginBtn" class="hidden px-4 py-2 border border-gray-400 rounded hover:bg-gray-100">Login</button>

      </div>
    </nav>

    <!-- Main content -->
    <main class="max-w-4xl mx-auto py-10 px-4">
      <form class="space-y-6" onsubmit="event.preventDefault(); uploadFile();">
        <div>
          <label for="functionName" class="block font-semibold mb-1">Importfunktion</label>
          <input type="text" id="functionName" value="api.import_projekt_row_call" required class="w-full border rounded px-3 py-2">
        </div>

        <div>
          <input type="file" id="fileInput" accept=".xlsx,.xls" class="mb-2">
          <div id="dropArea" class="bg-white border-2 border-dashed border-gray-300 rounded p-6 text-center text-gray-500">
            Excel-Datei hier ablegen oder auswählen
          </div>
          <div class="status text-sm mt-2 font-semibold" id="status"></div>
        </div>

        <div class="mb-8">
          <button type="button" onclick="toggleMapping()" class="text-blue-600 hover:underline font-medium mb-2 flex items-center gap-2 group">
              <svg id="mappingToggleIcon" xmlns="http://www.w3.org/2000/svg" class="w-5 h-5 text-blue-600 transition-transform duration-200 group-[.open]:rotate-180" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2">
                <path stroke-linecap="round" stroke-linejoin="round" d="M19 9l-7 7-7-7" />
              </svg>
              Feld-Mapping anzeigen/ausblenden
            </button>



          <div id="mappingSection" class="mt-4 bg-white border border-gray-200 rounded-xl shadow-sm p-4 hidden">
            <h3 class="font-semibold text-lg mb-4 text-gray-800">Feldzuordnung</h3>

            <div class="overflow-x-auto">
              <table class="min-w-full text-sm text-gray-700 border border-gray-200 rounded-xl overflow-hidden">
                <thead class="bg-gray-50 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">
                  <tr>
                    <th class="px-4 py-3 border">Feld (UI)</th>
                    <th class="px-4 py-3 border">DB-Feld</th>
                    <th class="px-4 py-3 border">Quelle</th>
                    <th class="px-4 py-3 border">Wert</th>
                  </tr>
                </thead>
                <tbody id="mappingTable" class="bg-white divide-y divide-gray-100">
                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Projektbezeichnung</td>
                    <td class="px-4 py-3 border"><code>pr_an_bez</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_an_bez" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Teamname" data-dbfield="pr_an_bez" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Projektnummer</td>
                    <td class="px-4 py-3 border"><code>pr_an_nr</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_an_nr" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Plannername" data-dbfield="pr_an_nr" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Übergeordnetes Arbeitspaket</td>
                    <td class="px-4 py-3 border"><code>pr_parent_ab_ap_nr</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_parent_ab_ap_nr" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Plannername" data-dbfield="pr_parent_ab_ap_nr" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Arbeitspaket</td>
                    <td class="px-4 py-3 border"><code>pr_ab_ap_nr</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_ab_ap_nr" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Bucket" data-dbfield="pr_ab_ap_nr" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Arbeitspaket Bezeichnung</td>
                    <td class="px-4 py-3 border"><code>pr_ab_ap_bem</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_ab_ap_bem" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Bucket" data-dbfield="pr_ab_ap_bem" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Kostenstelle</td>
                    <td class="px-4 py-3 border"><code>pr_a2_ks</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_a2_ks" class="source-select w-full border rounded px-2 py-1">
                        <option value="column">Excel-Spalte</option>
                        <option value="fixed" selected>Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="MON-PERS" data-dbfield="pr_a2_ks" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>

                  <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 border">Arbeitsgang</td>
                    <td class="px-4 py-3 border"><code>pr_a2_subject</code></td>
                    <td class="px-4 py-3 border">
                      <select data-dbfield="pr_a2_subject" class="source-select w-full border rounded px-2 py-1">
                        <option value="column" selected>Excel-Spalte</option>
                        <option value="fixed">Fester Wert</option>
                      </select>
                    </td>
                    <td class="px-4 py-3 border">
                      <input value="Aufgabe" data-dbfield="pr_a2_subject" class="value-input w-full border rounded px-2 py-1">
                    </td>
                  </tr>
                </tbody>
              </table>
            </div>
          </div>
        </div>

        <button type="button" onclick="toggleLog()" class="text-blue-600 hover:underline font-medium mb-2 flex items-center gap-2 group">
          <svg id="logToggleIcon" xmlns="http://www.w3.org/2000/svg" class="w-5 h-5 text-blue-600 transition-transform duration-200 group-[.open]:rotate-180" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2">
            <path stroke-linecap="round" stroke-linejoin="round" d="M19 9l-7 7-7-7" />
          </svg>
          Log anzeigen/ausblenden
        </button>

        <div id="logSection" class="mt-4 bg-white border border-gray-200 rounded-xl shadow-sm p-4 hidden">
          <h3 class="font-semibold text-lg mb-2 text-gray-800">Logausgabe</h3>
          <pre id="log" class="bg-gray-50 border border-gray-100 rounded p-3 text-sm text-gray-800 overflow-x-auto whitespace-pre-wrap max-h-96"></pre>
        </div>



        <div>
          <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white font-semibold px-6 py-2 rounded">
            Ausführen
          </button>
        </div>
      </form>
    </main>

    <!-- JavaScript-Teil wird nach Bedarf separat eingefügt oder erweitert -->
    <script>
    document.addEventListener('DOMContentLoaded', async () => {
      try {
        const { getCookie } = await import('./file?path=auth-header.js');
        const token = getCookie('jwt');
        const loginBtn = document.getElementById('loginBtn');
        const userMenu = document.getElementById('userDropdownWrap');
        const alwaysShowLogin = document.body.classList.contains('always-show-login');

        // Login-Button: nur wenn kein Token
        if (!token) {
          loginBtn?.classList.remove('hidden');
          loginBtn.style.display = 'inline-block';
        } else {
          loginBtn?.classList.add('hidden');
          loginBtn.style.display = 'none';
        }

        // Benutzerbereich: nur wenn always-show-login gesetzt und Token vorhanden
        if (alwaysShowLogin && token) {
          userMenu?.classList.remove('hidden');
          userMenu.style.display = 'flex';
        } else {
          userMenu?.classList.add('hidden');
          userMenu.style.display = 'none';
        }
      } catch (e) {
        console.error("Auth-Initialisierung fehlgeschlagen:", e);
      }
    });



      document.getElementById("loginBtn")?.addEventListener("click", () => {
        window.location.href = './login_page?redirect=importform';
      });

      document.getElementById("logoutDropdownBtn")?.addEventListener("click", () => {
        document.cookie = "jwt=; path=/; expires=Thu, 01 Jan 1970 00:00:00 GMT";
        window.location.href = './login_page?redirect=importform';
      });

      async function applyUserRights() {
        const { fetchWithAuth } = await import('./file?path=auth-header.js');
        const response = await fetchWithAuth('./execute_standard_sql_to_jsonb_set', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ standard_sql: 'Sondermodul.WebForm_Berechtigung' })
        });

        const data = await response.json();
        const user = data[0];
        const userNameSpan = document.getElementById("userName");
        const userAvatar = document.getElementById("userAvatar");

        if (!user) return;

        const initials = (user.ad_vorn?.charAt(0) || "") + (user.ad_name?.charAt(0) || "");
        userAvatar.textContent = initials || "U";
        userNameSpan.textContent = user.ad_vorn + " " + user.ad_name;
      }

      applyUserRights();

      // Dropdown öffnen/schließen bei Klick auf Avatar
        document.getElementById("userMenuBtn")?.addEventListener("click", function (e) {
          e.stopPropagation(); // verhindert Schließen durch document.click
          const dropdown = document.getElementById("userDropdown");
          dropdown.classList.toggle("opacity-0");
          dropdown.classList.toggle("pointer-events-none");
          dropdown.classList.toggle("scale-95");
        });

        // Schließen bei Klick außerhalb
        document.addEventListener("click", function () {
          const dropdown = document.getElementById("userDropdown");
          if (!dropdown.classList.contains("opacity-0")) {
            dropdown.classList.add("opacity-0", "pointer-events-none", "scale-95");
          }
        });

    // Drag & Drop, Mapping-Toggle, Log-Toggle, File-Upload

  document.addEventListener('DOMContentLoaded', () => {
    let selectedFile = null;

    const dropArea = document.getElementById('dropArea');
    const fileInput = document.getElementById('fileInput');
    const mappingSection = document.getElementById('mappingSection');
    const logSection = document.getElementById('logSection');

    function logMessage(msg) {
      const log = document.getElementById('log');
      if (log) log.textContent += msg + "\n";
    }

    function setStatus(message, isError = false) {
      const status = document.getElementById('status');
      if (status) {
        status.textContent = message;
        status.className = isError ? 'text-red-600' : 'text-green-600';
      }
    }

    function clearLog() {
      const log = document.getElementById('log');
      const status = document.getElementById('status');
      if (log) log.textContent = "";
      if (status) status.textContent = "";
    }

    function toggleMapping() {
      if (mappingSection) mappingSection.classList.toggle("hidden");
    }

    function toggleLog() {
      if (logSection) logSection.classList.toggle("hidden");
    }

    // Exponiere toggle-Funktionen im globalen Scope (weil sie im HTML direkt referenziert werden)
    window.toggleMapping = toggleMapping;
    window.toggleLog = toggleLog;

    if (dropArea) {
      dropArea.addEventListener('dragover', e => {
        e.preventDefault();
        dropArea.classList.add('bg-blue-50');
      });

      dropArea.addEventListener('dragleave', () => {
        dropArea.classList.remove('bg-blue-50');
      });

      dropArea.addEventListener('drop', e => {
        e.preventDefault();
        dropArea.classList.remove('bg-blue-50');
        selectedFile = e.dataTransfer.files[0];
        clearLog();
        setStatus("Datei per Drag & Drop ausgewählt: " + selectedFile.name);
      });
    }

    if (fileInput) {
      fileInput.addEventListener('change', () => {
        selectedFile = fileInput.files[0];
        clearLog();
        setStatus("Datei ausgewählt: " + selectedFile.name);
      });
    }

    // Upload-Funktion
    window.uploadFile = async function () {
      const { fetchWithAuth } = await import('./file?path=auth-header.js');
      const funcNameInput = document.getElementById('functionName');
      if (!selectedFile || !funcNameInput?.value.trim()) {
        alert("Bitte Importfunktion und Datei angeben.");
        return;
      }

      const funcName = funcNameInput.value.trim();

      const sourceSelectors = document.querySelectorAll('.source-select');
      const fieldMapping = {};

      sourceSelectors.forEach(selector => {
        const dbField = selector.getAttribute('data-dbfield');
        const row = selector.closest('tr');
        const input = row?.querySelector('.value-input');
        const value = input?.value.trim();
        const source = selector.value;
        if (dbField && value) {
          fieldMapping[dbField] = { source, value };
        }
      });

      logMessage("Gesendetes Mapping:\n" + JSON.stringify(fieldMapping, null, 2));

      const buffer = await selectedFile.arrayBuffer();
      const hex = Array.from(new Uint8Array(buffer)).map(b => b.toString(16).padStart(2, '0')).join('');
      const byteaHex = "\\x" + hex;

      const payload = {
        excel_file: byteaHex,
        import_func_name: funcName,
        field_mapping: fieldMapping
      };

      logMessage("Sende Anfrage ...");

      try {
        const response = await fetchWithAuth('http://localhost:3001/rpc/import_excel_via_function', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(payload)
        });

        const text = await response.text();
        const errorPattern = /fehler|error|exception/i;

        if (response.ok && !errorPattern.test(text)) {
          setStatus("Import erfolgreich.");
        } else {
          setStatus("Fehler beim Import!", true);
        }

        logMessage("Antwort [" + response.status + "]:\n" + text);
      } catch (err) {
        setStatus("Netzwerkfehler beim Import!", true);
        logMessage("Fehler: " + err.message);
      }
    };
  });

</script>

  </body>
</html>
$html$ );
$$ LANGUAGE sql;